/* $Id: arbsrc1.c,v 1.6 1998/09/16 02:04:54 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Arb source test program for one-shot and periodic modes */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: arbsrc1.c,v 1.6 1998/09/16 02:04:54 ericb Exp $";
static const char *progname;

static int
check_src_status(E1432ID hw, int nchan, SHORTSIZ16 *chan_list)
{
    int     i, status;

    for (i = 0; i < nchan; i++)
    {
	/* For these functions, negative return value means error,
	   positive means the condition happenned.  Either way, we
	   want to know about it. */

	status = e1432_check_src_overload(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_overload: returned %d\n",
			   progname, status);
	    return -1;
	}

	status = e1432_check_src_overread(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_overread: returned %d\n",
			   progname, status);
	    return -1;
	}

	status = e1432_check_src_shutdown(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_shutdown: returned %d\n",
			   progname, status);
	    return -1;
	}

	status = e1432_check_src_underrun(hw, chan_list[i]);
	if (status != 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_underrun: returned %d\n",
			   progname, status);
	    return -1;
	}
    }

    return 0;
}

static char *
state_to_str(LONGSIZ32 state)
{
    if (state == E1432_SRCBUF_AVAIL)
	return "Avail ";
    else if (state == E1432_SRCBUF_RDY)
	return "Rdy   ";
    else if (state == E1432_SRCBUF_FULL)
	return "Full  ";
    else
	return "Error!";
}

static int
check_src_arbstates(E1432ID hw, int nchan, SHORTSIZ16 *chan_list, int verbose)
{
    LONGSIZ32 words, ab, a, b;
    int     i;

    for (i = 0; i < nchan; i++)
    {
	CHECK(e1432_get_src_arbstates(hw, chan_list[i], &words, &ab, &a, &b));
	if (b != E1432_SRCBUF_FULL)
	{
	    (void) fprintf(stderr, "%s: B state %s\n",
			   progname, state_to_str(b));
	    return -1;
	}
	if (verbose)
	    (void) printf("chan %4d, words %6ld, ab %s, a %s, b %s\n",
			  chan_list[i], words, state_to_str(ab),
			  state_to_str(a), state_to_str(b));
    }

    return 0;
}

static int
check_arbrdy(E1432ID hw, int nchan, SHORTSIZ16 *chan_list)
{
    int     i, status;

    for (i = 0; i < nchan; i++)
    {
	CHECK(status = e1432_check_src_arbrdy(hw, chan_list[i],
					      E1432_SRC_DATA_MODE_WAITA));
	if (status == 0)
	    return 0;
    }
    return 1;
}

static int
wait_arbrdy(E1432ID hw, int chan)
{
    clock_t start, timeout;
    int     status;

    timeout = 2 * CLOCKS_PER_SEC;
    start = clock();
    while ((status = e1432_check_src_arbrdy(hw, chan,
					    E1432_SRC_DATA_MODE_WAITA)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_check_src_arbrdy(hw, chan,
					     E1432_SRC_DATA_MODE_WAITA)) == 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_check_src_arbrdy: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_check_src_arbrdy: returned %d\n",
		       progname, status);
	return -1;
    }

    return 0;
}

/* Fill one source buffer, either A or B, for all source channels we
   are using. */
static int
xfer_src_data(E1432ID hw, int nchan, SHORTSIZ16 *chan_list,
	      LONGSIZ32 *buf, long xfersize, long sbsize)
{
    long    xfer, offset, npoint;
    int     i;

    /* We must loop through all channels doing only xfersize words at
       a time. */
    offset = 0;
    npoint = sbsize;
    while (npoint > 0)
    {
	xfer = xfersize;
	if (xfer > npoint)
	    xfer = npoint;

	for (i = 0; i < nchan; i++)
	{
	    CHECK(wait_arbrdy(hw, chan_list[i]));
	    CHECK(e1432_write_srcbuffer_data(hw, chan_list[i],
					     buf + offset, xfer,
					     E1432_SRC_DATA_MODE_WAITA));
	}

	npoint -= xfer;
	offset += xfer;
    }

    return 0;
}

static int
init(SHORTSIZ16 laddr, E1432ID *hw, int *group,
     int ch1, int ch2, int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig;
    int     limit;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(1, &laddr, hw));
    CHECK(e1432_get_hwconfig(1, &laddr, &hwconfig));

    limit = ch2 ? 2 : 1;
    if (hwconfig.source_chans < limit)
    {
	(void) fprintf(stderr, "%s: Need %d source channel(s)\n",
		       progname, limit);
	return -1;
    }

    *nchan = 0;
    if (ch1)
	chan_list[(*nchan)++] = E1432_SOURCE_CHAN(1);
    if (ch2)
	chan_list[(*nchan)++] = E1432_SOURCE_CHAN(2);

    *group = e1432_create_channel_group(*hw, *nchan, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group, int mode, long sbsize)
{
    CHECK(e1432_set_active(hw, group, E1432_CHANNEL_ON));
    CHECK(e1432_set_clock_freq(hw, group, 51200));
    CHECK(e1432_set_ramp_rate(hw, group, 0));
    CHECK(e1432_set_range(hw, group, 1));
    CHECK(e1432_set_source_mode(hw, group, E1432_SOURCE_MODE_ARB));
    CHECK(e1432_set_span(hw, group, 20000));
    CHECK(e1432_set_srcbuffer_mode(hw, group, mode));
    CHECK(e1432_set_srcbuffer_size(hw, group, sbsize));
    CHECK(e1432_set_srcbuffer_init(hw, group, E1432_SRCBUFFER_INIT_EMPTY));

    return 0;
}

static long
src_bits(double x)
{
    if (x > (double) 0x7fffffL / 0x800000L)
	x = (double) 0x7fffffL / 0x800000L;
    else if (x < -1.0)
	x = -1.0;
    return (long) floor(8388608 * x + 0.5) << 8;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    long xfersize, long sbsize, int verbose)
{
    LONGSIZ32 *buf;
    long    i;
    int     status;
    double  freq, phase;

    buf = malloc(sbsize * sizeof *buf);
    if (buf == NULL)
    {
	(void) fprintf(stderr, "%s: malloc failed\n", progname);
	return -1;
    }

    /* Create swept sine, 0 to 20 kHz */
    phase = 0;
    freq = 0;
    for (i = 0; i < sbsize; i++)
    {
	buf[i] = src_bits(sin(phase));
	freq += 20000.0 / sbsize;
	phase += freq * 2 * M_PI / 51200;
    }

#if 0
    /* Put signal only at the end, to see that the end gets flushed
       out properly. */
    (void) memset(buf, 0, sizeof buf);
    if (sbsize > 8000)
	i = sbsize - 4000;
    else
	i = sbsize / 2;
    for (; i < sbsize; i++)
    {
	if ((i & 0x1) == 0)
	    buf[i] = 0x7fffff00;
	else
	    buf[i] = 0x80000000;
    }
#endif

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list));
    CHECK(check_src_arbstates(hw, nchan, chan_list, verbose));
    CHECK(status = check_arbrdy(hw, nchan, chan_list));
    if (nchan > 0 && status == 0)
    {
	(void) fprintf(stderr,
		       "%s: no arbrdy before sending A data?\n",
		       progname);
	return -1;
    }

    /* Fill A */
    if (verbose)
	(void) printf("Xfer scan 1 (A buffer)\n");
    CHECK(xfer_src_data(hw, nchan, chan_list, buf, xfersize, sbsize));

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list));
    CHECK(check_src_arbstates(hw, nchan, chan_list, verbose));
    CHECK(status = check_arbrdy(hw, nchan, chan_list));
    if (nchan > 0 && status != 0)
    {
	(void) fprintf(stderr,
		       "%s: arbrdy after sending A data before init measure?\n",
		       progname);
	return -1;
    }

    (void) sleep(1);

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list));
    CHECK(check_src_arbstates(hw, nchan, chan_list, verbose));
    CHECK(status = check_arbrdy(hw, nchan, chan_list));
    if (nchan > 0 && status != 0)
    {
	(void) fprintf(stderr,
		       "%s: arbrdy after sending A data before init measure?\n",
		       progname);
	return -1;
    }

    /* Start measurement */
    if (verbose)
	(void) printf("e1432_init_measure\n");
    CHECK(e1432_init_measure(hw, group));

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list));
    CHECK(check_src_arbstates(hw, nchan, chan_list, verbose));
    CHECK(status = check_arbrdy(hw, nchan, chan_list));
    if (nchan > 0 && status != 0)
    {
	(void) fprintf(stderr,
		       "%s: arbrdy after init measure?\n",
		       progname);
	return -1;
    }

    (void) sleep(1);

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list));
    CHECK(check_src_arbstates(hw, nchan, chan_list, verbose));
    CHECK(status = check_arbrdy(hw, nchan, chan_list));
    if (nchan > 0 && status != 0)
    {
	(void) fprintf(stderr,
		       "%s: arbrdy after init measure?\n",
		       progname);
	return -1;
    }

    (void) sleep(sbsize / 51200);

    /* Check source status */
    CHECK(check_src_status(hw, nchan, chan_list));
    CHECK(check_src_arbstates(hw, nchan, chan_list, verbose));
    CHECK(status = check_arbrdy(hw, nchan, chan_list));
    if (nchan > 0 && status != 0)
    {
	(void) fprintf(stderr,
		       "%s: arbrdy after init measure?\n",
		       progname);
	return -1;
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-12uvV] [-L laddr] [-m mode]\n"
		   "\t     [-s sbsize] [-x xfersize]\n"
		   "\t-1: Use source channel 1\n"
		   "\t-2: Use source channel 2\n"
		   "\t-L: Logical address is <laddr>, default 8\n"
		   "\t-m: Source buffer mode <mode>, default oneshot\n"
		   "\t-s: Source buffer size <sbsize>, default 20480\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n"
		   "\t-x: Source transfer size <xfersize>, default 4096\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr;
    SHORTSIZ16 chan_list[2];
    char   *p;
    int     c, ch1, ch2, mode, verbose, group, nchan;
    long    xfersize, sbsize;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    ch1 = 0;
    ch2 = 0;
    laddr = 8;
    mode = E1432_SRCBUFFER_ONESHOT;
    sbsize = 20480;
    verbose = 0;
    xfersize = E1432_SRC_DATA_NUMWORDS_MAX;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "12L:m:s:uvVx:")) != -1)
	switch (c)
	{
	case '1':
	    ch1 = 1;
	    break;
	case '2':
	    ch2 = 1;
	    break;
	case 'L':
	    laddr = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr < 0 || laddr > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'm':
	    if (strcmp(optarg, "oneshot") == 0)
		mode = E1432_SRCBUFFER_ONESHOT;
	    else if (strcmp(optarg, "periodic") == 0)
		mode = E1432_SRCBUFFER_PERIODIC_A;
	    else
	    {
		(void) fprintf(stderr,
			       "%s: invalid srcbuffer mode: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 's':
	    sbsize = strtol(optarg, &p, 0);
	    if (optarg == p || sbsize < 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid source buffer size: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'x':
	    xfersize = strtol(optarg, &p, 0);
	    if (optarg == p || xfersize <= 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid source transfer size: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    if (!ch1 && !ch2)
    {
	(void) fprintf(stderr, "%s: activate at least one source channel\n",
		       progname);
	usage();
    }

    /* Run the measurement */
    if (init(laddr, &hw, &group, ch1, ch2, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, mode, sbsize) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list, xfersize, sbsize, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
